#library(mva)
#source("SLprin.R")
#load("SimulatedAmounts.RData")
# Datamatrices and data.frames and vectors should be given as a classless
# vector or matrix
gsi.plain <- function(x) {
  if( is.data.frame(x) )
    unclass(data.matrix(x))
  else 
    unclass(x)
}

gsi.simshape <- function(x,oldx) {
  if(length(dim(oldx))>=2 )
    oneOrDataset(x)
  else if( length(dim(oldx)) == 1 )
    structure(c(x),dim=length(x))
  else 
    c(drop(x))
}

gsi.diagExtract <- function(x) {
  if( length(x) > 1 )
    diag(x)
  else
    c(x)
}

gsi.diagGenerate <- function(x) {
  if( length(x) > 1 )
    diag(x)
  else
    matrix(x)
}

gsi.getD  <- function(x) ncol(oneOrDataset(x))
gsi.getN  <- function(x) nrow(oneOrDataset(x))   


clo <- function(X,parts=1:NCOL(oneOrDataset(X)),total=1) {
  X <- gsi.plain(X)
  parts  <- unique(parts)
  if( is.character(parts) ) {
    partsn <- match(parts,colnames(X))
    if( any(is.na(partsn)) )
      stop("Unknown variable name",d[is.na(partsn)])
    parts <- partsn
  }
  nparts <- length(parts)
  Xn <- gsi.plain(oneOrDataset(X))[,parts,drop=FALSE]
  drop <- length(dim(X)) < 2
  if( any(na.omit(c(Xn)<0)) )
    stop("Negative values are not valid for amounts")
  nas <- is.na(c(Xn))
  if( !is.na(total) ) {
    Xn[nas]<-0
    s <- c(Xn %*% rep(1,nparts))
    Xn  <- Xn / matrix(rep(s/total,nparts),ncol=nparts)
    Xn[nas] <- NA
  }
  gsi.simshape(Xn,X)
}

names.acomp <- function(x) colnames(oneOrDataset(x))
names.rcomp <- names.acomp
names.aplus <- names.acomp
names.rplus <- names.acomp
names.rmult <- names.acomp

groupparts <- function(x,...) UseMethod("groupparts",x)

groupparts.rcomp <- function(x,...,groups=list(...)) {
  x <- rmult(clo(x))
  usedparts <- unique(unlist(lapply(groups,function(i) {
    if( is.character(i) ) {
      parts <- match(i,names(x))
      if( any(is.na(parts)))
        stop("Unknown part",i[is.na(parts)])
      parts
    } else i
  })))
  otherparts <- (1:gsi.getD(x))[-usedparts]
  if( length(otherparts) >0 ) {
    names(otherparts) <- names(x)[otherparts]
    groups <- c(groups,otherparts)
  }
  rcomp( sapply(groups,function(idx) {
    ss <- rplus(x,idx)
    ss %*% rep(1,gsi.getD(ss))
  }))
}

groupparts.rplus <- function(x,...,groups=list(...)) {
  x <- rmult(x)
  usedparts <- unique(unlist(lapply(groups,function(i) {
    if( is.character(i) ) {
      parts <- match(i,names(x))
      if( any(is.na(parts)))
        stop("Unknown part",i[is.na(parts)])
      parts
    } else i
  })))
  otherparts <- (1:gsi.getD(x))[-usedparts]
  if( length(otherparts) >0 ) {
    names(otherparts) <- names(x)[otherparts]
    groups <- c(groups,otherparts)
  }
  rplus( sapply(groups,function(idx) {
    ss <- rplus(x,idx)
    ss %*% rep(1,gsi.getD(ss))
  }))
}

groupparts.acomp <- function(x,...,groups=list(...)) {
  x <- rmult(x)
  usedparts <- unique(unlist(lapply(groups,function(i) {
    if( is.character(i) ) {
      parts <- match(i,names(x))
      if( any(is.na(parts)))
        stop("Unknown part",i[is.na(parts)])
      parts
    } else i
  })))
  otherparts <- (1:gsi.getD(x))[-usedparts]
  if( length(otherparts) >0 ) {
    names(otherparts) <- names(x)[otherparts]
    groups <- c(groups,otherparts)
  }
  acomp( sapply(groups,function(idx) {
    ss <- aplus(x,idx)
    if( is.matrix(ss) )
      geometricmean.row(ss)
    else
      geometricmean(ss)
  }))
}

groupparts.aplus <- function(x,...,groups=list(...)) {
  x <- rmult(x)
  usedparts <- unique(unlist(lapply(groups,function(i) {
    if( is.character(i) ) {
      parts <- match(i,names(x))
      if( any(is.na(parts)))
        stop("Unknown part",i[is.na(parts)])
      parts
    } else i
  })))
  otherparts <- (1:gsi.getD(x))[-usedparts]
  if( length(otherparts) >0 ) {
    names(otherparts) <- names(x)[otherparts]
    groups <- c(groups,otherparts)
  }
  aplus( sapply(groups,function(idx) {
    ss <- aplus(x,idx)
    if( is.matrix(ss) )
      geometricmean.row(ss)
    else
      geometricmean(ss)
  }))
}



# groupparts(x,G1=c("Cd","S"),G2=c("Co","Ni"),G3=c("As","F"))

acomp <- function(X,parts=1:NCOL(oneOrDataset(X)),total=1) {
    X <-  structure(clo(X,parts,total),class="acomp")
    if( any(na.omit(c(X)<=0)) )
      warning("Compositions has nonpositiv values")
    X
}

rcomp <- function(X,parts=1:NCOL(oneOrDataset(X)),total=1) {
    X <-  structure(clo(X,parts,total),class="rcomp")
    X
}


aplus <- function(X,parts=1:NCOL(oneOrDataset(X)),total=NA) {
  X <- gsi.simshape(clo(X,parts,total),X)
  if( any(na.omit(c(X)<0)) )
    stop("Negativ values in aplus")
  if( any(na.omit(c(X)<=0)) )
    warning("Not all values positiv in aplus")
  class(X) <-"aplus"
  X
}

rplus <- function(X,parts=1:NCOL(oneOrDataset(X)),total=NA) {
  X <- gsi.simshape(clo(X,parts,total),X)
  if( any(na.omit(c(X)<0)) )
    stop("Negativ values in rplus")
  class(X) <-"rplus"
  X
}

rmult <- function(X,parts=1:NCOL(oneOrDataset(X))) {
  X <- gsi.simshape(oneOrDataset(X)[,parts,drop=FALSE],X)
  class(X) <-"rmult"
  X
}



gsi2.invperm <- function(i,n){
  i <- unique(c(i,1:n))
  j <- numeric(length(i))
  j[i]<-1:length(i)
  j
}


rcompmargin <- function(X,d=c(1,2),name="+",pos=length(d)+1) {
  X <- rcomp(X)
  drop <- length(dim(X)) < 2
  if( mode(d)=="character" )
    d <- match(d,colnames(X))
  X <- oneOrDataset(gsi.plain(X))
  d <- unique(d)
  if( NCOL(X) <= length(d) )
    return(rcomp(X))
  else if( NCOL(X) == length(d) +1)
    return( rcomp(cbind(X[,d,drop=FALSE],X[,-d,drop=FALSE]) ))
  Xm <- X[,-d,drop=FALSE]
  tmp <- rcomp(cbind(Rest=Xm %*% rep(1,NCOL(Xm)) ,X[,d,drop=FALSE] ))
  if( !is.null(colnames(tmp)) )
    colnames(tmp)[1]<-name
  if( pos != 1 )
    tmp <- tmp[,gsi2.invperm(pos,ncol(tmp))]
  if( drop )
    tmp <- drop(tmp)
  rcomp(tmp)
}

acompmargin <- function(X,d=c(1,2),name="*",pos=length(d)+1) {
  drop <- length(dim(X)) < 2
  if( mode(d)=="character" )
    d <- match(d,colnames(X))
  X <- oneOrDataset(gsi.plain(X))
  d <- unique(d)
  if( NCOL(X) <= length(d) )
    return(X)
  else if( NCOL(X) == length(d) +1)
    return( cbind(X[,d,drop=FALSE],X[,-d,drop=FALSE]) )
  Xm <- X[,-d]
  tmp <- acomp(cbind(Rest=exp(log(Xm) %*% rep(1/NCOL(Xm),NCOL(Xm))) ,X[,d,drop=FALSE] ))
  if( !is.null(colnames(tmp)) )
    colnames(tmp)[1]<-name
  if( pos != 1 )
    tmp <- tmp[,gsi2.invperm(pos,ncol(tmp))]
  if( drop )
    tmp <- drop(tmp)
  acomp(tmp)
}


oneOrDataset <- function(W,B=NULL) {
  W <- gsi.plain(W)
  if( missing(B) || length(dim(B))!= 2 ) {
    if( length(dim(W)) == 2) {
      return( W )
    }
    else {
      tmp <- matrix(c(W),nrow=1)
      colnames(tmp) <- names(W)
      return(tmp)
    }
  } else {
    if( length(dim(W)) == 2) {
      return( W )
    }
    else {
      tmp <- matrix(c(W),nrow=NROW(B),ncol=length(W),byrow=TRUE)
      colnames(tmp)<- names(W)
      return(tmp)
    }
  }
}



geometricmean <- function(x,...) { exp(mean(log(c(unclass(x))),...)) }

geometricmean.row <- function(x,...) apply(x,1,geometricmean,...)
geometricmean.col <- function(x,...) apply(x,2,geometricmean,...)

mean.col <- function( x , ... , na.action=get(getOption("na.action"))) {
  apply(na.action(oneOrDataset(x)),2,mean,...)
}

mean.row <- function( x , ... , na.action=get(getOption("na.action"))) {
  apply(na.action(oneOrDataset(x)),1,mean,...)
}


totals <- function( x , ... ) UseMethod("totals",x)

totals.acomp <- function(x,...) {
apply(na.action(oneOrDataset(x)),1,sum,...)
}

totals.rcomp <- totals.acomp
totals.aplus <- totals.acomp
totals.rplus <- totals.acomp


mean.acomp <- function( x,..., na.action=get(getOption("na.action")) ) {
  clr.inv(mean.col(clr(na.action(x)),...))
}

mean.rcomp <- function( x,..., na.action=get(getOption("na.action")) ) {
  cpt.inv(mean.col(cpt(na.action(x)),..., na.action=get(getOption("na.action"))))
}

mean.aplus <- function( x,..., na.action=get(getOption("na.action")) ) {
  ilt.inv(mean.col(ilt(na.action(x)),...))
}

mean.rplus <- function( x,..., na.action=get(getOption("na.action")) ) {
  iit.inv(mean.col(iit(na.action(x)),...))
}

mean.rmult <- function( x,..., na.action=get(getOption("na.action")) ) {
  rmult(mean.col(unclass(na.action(x)),...))
}


var         <- function(x,...) UseMethod("var",x)
var.default <- base::var
formals(var.default) <- c(formals(var.default),alist(...= ))
var.acomp   <- function(x,y=NULL,...) {
  var(cdt(x),cdt(y),...)
}
var.rcomp <- var.acomp
var.aplus <- var.acomp
var.rplus <- var.acomp
var.rmult <- function(x,y=NULL,...) {
  var(unclass(x),unclass(cdt(y)),...)
}

cov         <- function(x,y=x,...) UseMethod("cov",x)
cov.default <- base::cov
formals(cov.default) <- c(formals(cov.default),alist(...= ))
cov.acomp   <- function(x,y=NULL,...) {
  cov(cdt(x),cdt(y),...)
}
cov.rcomp <- cov.acomp
cov.aplus <- cov.acomp
cov.rplus <- cov.acomp
cov.rmult <- function(x,y=NULL,...) {
  cov(unclass(x),unclass(cdt(y)),...)
}


clr2ilr <- function( x , V=ilrBase(x) ) {
  gsi.simshape( oneOrDataset(x) %*% V , x)
}

ilr2clr <- function( z , V=ilrBase(z=z) ) {
  gsi.simshape( oneOrDataset(z) %*% t(V) , z)
}


clrvar2ilr <- function( varx , V=ilrBase(D=ncol(varx)) ) {
  t(V) %*% varx %*% V
}

ilrvar2clr <- function( varz , V=ilrBase(D=ncol(varz)+1) ) {
  V %*% varz %*% t(V)
}

powerofpsdmatrix <- function(M,p,...) {
  s <- svd(M,...)
  d <- ifelse( abs(s$d)>max(abs(s$d))*1E-10, s$d,0)
  s$u %*% gsi.diagGenerate(d^p) %*% t(s$v)
}

mvar <- function(x,...) UseMethod("mvar",x)
mcov <- function(x,...) UseMethod("mcov",x)
mcor <- function(x,...) UseMethod("mcor",x)
msd  <- function(x,...) UseMethod("msd",x)

mvar.default <- function(x,y=NULL,...) {
  sum(gsi.diagExtract(var(x,y,...)))
}

mcov.default <- function(x,y=x,...) {
  sum(abs(svd(cov(idt(x),idt(y),...))$d))
}

msd.default <- function(x,y=NULL,...) {
  sqrt(mean(gsi.diagExtract(var(idt(x),y=NULL,...))))
}

mcor.default <- function(x,y,...) {
  ix <- scale(idt(x),center=TRUE,scale=FALSE)
  ix <- ix %*% powerofpsdmatrix(var(ix),-1/2)
  iy <- scale(idt(y),center=TRUE,scale=FALSE)
  iy <- iy %*% powerofpsdmatrix(var(iy),-1/2)
  mcov(ix,iy)
}


summary.acomp <- function( object,... ) {
  W <- clo(gsi.plain(object))
  Wq <- apply(W,1,function(w) outer(w,w,"/"))
  dim(Wq)<-c(ncol(W),ncol(W),nrow(W))
  dimnames(Wq) <- list(colnames(W),colnames(W),NULL)
  structure(list(mean=mean(acomp(W)),
       mean.ratio=apply(Wq,1:2,function(x) exp(mean(log(x)))),
       variation=variation.acomp(acomp(W)),
       expsd=exp(sqrt(variation.acomp(acomp(W)))),
       min=apply(Wq,1:2,min),
       q1 =apply(Wq,1:2,quantile,probs=0.25),
       med=apply(Wq,1:2,median),
       q3 =apply(Wq,1:2,quantile,probs=0.75),
       max=apply(Wq,1:2,max)
       ),class="summary.acomp")
       
}

summary.aplus <- function( object,...,digits=max(3, getOption("digits")-3)  ) {
  object <- ilt(object)
  erg <- sapply(data.frame(object),summary,...,digits=18)
  erg <- apply(erg,1:2,exp)
  erg <- apply(erg,1:2,signif,digits=digits)
  class(erg) <- c("summary.aplus",class(erg))
  erg       
}

summary.rplus <- function( object,...  ) {
  object <- iit(object)
  erg <- sapply(data.frame(object),summary,...)
  class(erg) <- c("summary.rplus",class(erg))
  erg       
}

summary.rmult <- function( object,...  ) {
  object <- unclass(object)
  erg <- sapply(data.frame(object),summary,...)
  class(erg) <- c("summary.rmult",class(erg))
  erg       
}

summary.rcomp <- function( object,...) {
  object <- clo(gsi.plain(object)) 
  erg <- sapply(data.frame(object),summary,...)
  erg <- apply(erg,1:2,exp)
  class(erg) <- c("summary.rcomp",class(erg))
  erg       
}



vp.boxplot <- function(x,y,...,dots=FALSE,boxes=TRUE,xlim,ylim,log,notch=FALSE) {
    if( boxes ) boxplot(split(y,x),add=TRUE,notch=notch)
    if( dots  ) points(x,y,...)
}

gsi.textpanel <- function(x,y,lab,...) {
  par(usr=c(0,1,0,1),xlog=FALSE,ylog=FALSE)
  text(0.5,0.5,lab,...)
}

boxplot.acomp <- function(x,fak=NULL,...,
                         xlim=x.lim,ylim=c(minq,maxq),log=TRUE,panel=vp.boxplot,dots=!boxes,boxes=TRUE) {
  X <- acomp(x)
  if( is.null(fak) )
    fak <- factor(rep("",nrow(X)))
  if( is.factor(fak) )
    x.lim <- c(0,nlevels(fak)+1)
  else {
    x.lim <- range(fak)
    boxes <- F
    dots  <- T
  }
  if( is.function(panel) )
    panel <- list(panel)
  ipanel <- function(x,y,...) {
    a <- unclass(X)[,gsi.mapfrom01(log(x))]
    b <- unclass(X)[,gsi.mapfrom01(log(y))]
    for( thispanel in panel ) 
      thispanel(fak,b/a,...,dots=dots,boxes=boxes)
  }
  su <- summary.acomp(X)
  minq <- min(su$min)
  maxq <- max(su$max)
  print(colnames(X))
  mm <- exp(sapply(1:NCOL(X),gsi.mapin01))
  colnames(mm) <- colnames(X)
  pairs(mm,labels=colnames(X),panel=ipanel,...,log=ifelse(log,"y",""),ylim=ylim,xlim=xlim,text.panel=gsi.textpanel)
  

}

boxplot.rcomp <- function(x,fak=NULL,...,
                         xlim=x.lim,ylim=c(0,1),log=FALSE,panel=vp.boxplot,dots=!boxes,boxes=TRUE) {
  X <- acomp(x)
  if( is.null(fak) )
    fak <- factor(rep("",nrow(X)))
  if( is.factor(fak) )
    x.lim <- c(0,nlevels(fak)+1)
  else {
    x.lim <- range(fak)
    boxes <- F
    dots  <- T
  }
  if( is.function(panel) )
    panel <- list(panel)
  if( missing(ylim) && log )
    ylim <- c(minq,maxq)
  ipanel <- function(x,y,...) {
    a <- unclass(X)[,gsi.mapfrom01(log(x))]
    b <- unclass(X)[,gsi.mapfrom01(log(y))]
    for( thispanel in panel ) 
      thispanel(fak,if(log) log(b/(a+b)) else b/(a+b),
                ...,dots=dots,boxes=boxes)
  }
  su <- summary.acomp(X)
  minq <- log(min(su$min)/(min(su$min)+max(su$max)))
  maxq <- log(max(su$max)/(min(su$min)+max(su$max)))
  pairs(exp(sapply(1:NCOL(X),gsi.mapin01)),labels=colnames(X),panel=ipanel,...,ylim=ylim,xlim=xlim)
  

}


vp.qqnorm <- function(x,y,...,alpha=NULL) {
  usr <- par("usr")
  usr[1:2] <- range(qnorm(ppoints(length(y))))
  usr[3:4] <- range(y)
  par( usr=usr )
  if( !is.null(alpha) && is.factor(x) ) 
    alpha <- alpha/nlevels(x)
  reject <- F
  if( is.factor(x)) {
    for( k in split(y,x) ) {
      if( !is.null(alpha) && shapiro.test(k)$p < alpha )
        reject<-T
      lines(qnorm(ppoints(length(k))),sort(k),...)
    }
  } else { 
    if( !is.null(alpha) && shapiro.test(y)$p < alpha )
        reject<-T
    points(qnorm(ppoints(length(y))),sort(y),...)
  }
  qqline(y)
  if( reject )
    title(main="!",col.main="red")
    
}

qqnorm.acomp <- function(y,fak=NULL,...,panel=vp.qqnorm,alpha=NULL) {
  X <- acomp(y)
  if( !is.null(alpha) )
    alpha <- alpha/((nrow(X)*(nrow(X)-1)/2))
  if( is.function(panel) )
    panel <- list(panel)
  ipanel <- function(x,y,...) {
    a <- unclass(X)[,gsi.mapfrom01(x)]
    b <- unclass(X)[,gsi.mapfrom01(y)]
    v <- log(b/a)
    for( thispanel in panel )
      thispanel(fak,v,...,alpha=alpha)
  }
  pairs(sapply(1:NCOL(X),gsi.mapin01),labels=colnames(X),panel=ipanel,...)
}

qqnorm.rcomp <- function(y,fak=NULL,...,panel=vp.qqnorm,alpha=NULL) {
  X <- acomp(y)
  if( is.function(panel) )
    panel <- list(panel)
  if( !is.null(alpha) )
    alpha <- alpha/(nrow(X)^2)
  ipanelupper <- function(x,y,...) {
    a <- unclass(X)[,gsi.mapfrom01(x)]
    b <- unclass(X)[,gsi.mapfrom01(y)]
    for( thispanel in panel )
      thispanel(fak,b-a,...,alpha=alpha)
  }
  ipanellower <- function(x,y,...) {
    a <- unclass(X)[,gsi.mapfrom01(x)]
    b <- unclass(X)[,gsi.mapfrom01(y)]
    for( thispanel in panel )
      thispanel(fak,b+a,...,alpha=alpha)
  }
  ipaneldiag <- function(x,...) {
    a <- unclass(X)[,gsi.mapfrom01(x)]
    for( thispanel in panel )
      thispanel(fak,a,...,alpha=alpha)
  }
  itextpanel <- function(x,y,lab,...) {
    par(usr=c(0,1,0,1),xlog=FALSE,ylog=FALSE)
    text(0.1,0.9,lab,adj=c(0,1),...)
  }

  pairs(sapply(1:NCOL(X),gsi.mapin01),labels=colnames(X),lower.panel=ipanellower,upper.panel=ipanelupper,diag.panel=ipaneldiag,text.panel=itextpanel,...)
}


gsi.drop  <-function(X,drop) if( drop ) drop(X) else X

is.acomp <- function(x) inherits(x,"acomp")

is.rcomp <- function(x) inherits(x,"rcomp")

is.aplus <- function(x) inherits(x,"aplus")

is.rplus <- function(x) inherits(x,"rplus")
   
is.rmult <- function(x) inherits(x,"rmult")


perturbe <- function( x,y ) {
  acomp(gsi.mul(x,y))
}

perturbe.aplus <- function(x,y) {
  aplus(gsi.mul(x,y))
}



gsi.add <- function( x,y ) {
  if( length(dim(x)) == 2 )
    if( length(dim(y)) == 2 )
      unclass(x)+unclass(y)
    else
      unclass(x)+rep(c(y),rep(NROW(x),length(y)))
  else if( length(dim(y)) == 2 )
      unclass(y)+rep(c(x),rep(NROW(y),length(x)))
  else
    unclass(x)+unclass(y)
}

gsi.sub <- function( x,y ) {
 # drop <- length(dim(x)) < 2 && length(dim(y)) < 2
  if( length(dim(x)) == 2 )
    if( length(dim(y)) == 2 )
      unclass(x)-unclass(y)
    else
      unclass(x)-rep(c(y),rep(NROW(x),length(y)))
  else if( length(dim(y)) == 2 )
      unclass(y)-rep(c(x),rep(NROW(y),length(x)))
  else
    unclass(x)-unclass(y)
}

gsi.mul <- function( x,y ) {
  if( length(dim(x)) == 2 )
    if( length(dim(y)) == 2 )
      unclass(x)*unclass(y)
    else
      unclass(x)*rep(c(y),rep(NROW(x),length(y)))
  else if( length(dim(y)) == 2 )
      unclass(y)*rep(c(x),rep(NROW(y),length(x)))
  else
    unclass(x)*unclass(y)
}

gsi.div <- function( x,y ) {
  if( length(dim(x)) == 2 )
    if( length(dim(y)) == 2 )
      unclass(x)/unclass(y)
    else
      unclass(x)/rep(c(y),rep(NROW(x),length(y)))
  else if( length(dim(y)) == 2 )
      unclass(y)/rep(c(x),rep(NROW(y),length(x)))
  else
    unclass(x)/unclass(y)
}


power.acomp <- function(x,s) {
  if( is.acomp(s) || is.rcomp(s))
    stop("power.acomp is scalar multiplication only")
  if( !is.matrix(x) || nrow(x)==1 ) {
    if( length(s)>1 )
      x <- matrix(x,byrow=T,ncol=length(x),nrow=length(s))
  } else {
    if( length(s) > 1 && length(s)!= nrow(x) )
      warning("lengths don't match in power.acomp")
  }
  acomp(unclass(x)^c(s)) 
}


"+.acomp" <- function(x,y) {
  acomp(gsi.mul(x,y))
}

"-.acomp" <- function(x,y) {
  if( missing(y) )
    acomp(1/unclass(x))
  else 
    acomp(gsi.div(x,y))
}

"*.acomp" <- function(x,y) {
  if( is.acomp(x) && !is.acomp(y) )
    power.acomp(x,y)
  else if( is.acomp(y)&& !is.acomp(x) )
    power.acomp(y,x)
  else
    stop("the powertransform performed in *.acomp only operates on acomps and scalar")
}

"/.acomp" <- function(x,y) {
  if( is.acomp(x) && !is.acomp(y) )
    power.acomp(x,1/unclass(y))
  else
    stop("/.acomp only operates on acomp / numeric")
}

"+.aplus" <- function(x,y) {
    aplus(gsi.mul(x,y))
}

"-.aplus" <- function(x,y) {
  if( missing(y) )
    return(aplus(1/unclass(y)))
  else
    aplus( gsi.div(x,y) )
}

"*.aplus" <- function(x,y) {
  if( is.aplus(x)&& !is.aplus(y) )
    power.aplus(x,y)
  else if( is.aplus(y)&& !is.aplus(x) )
    power.aplus(y,x)
  else
    stop("*.aplus only operates on aplus and scalar")
}

"/.aplus" <- function(x,y) {
  if( is.aplus(x) && !is.aplus(y) )
    power.aplus(x,1/unclass(y))
  else
    stop("/.aplus only operates on aplus and scalar")
}


"+.rcomp" <- function(x,y) {
  warning("+ is meaningless for rcomp")
  rcomp(gsi.add(x,y))
}

"-.rcomp" <- function(x,y) {
  if( missing(y) )
    rmult(-unclass(x))
  else
    rmult(gsi.sub(x,y))
}

"*.rcomp" <- function(x,y) {
  if( is.rcomp(x) && is.rcomp(y) )
    rcomp(gsi.mul(x,y))
  else if( is.rcomp(x) )
    rplus(x)*y
  else if( is.rcomp(y) )
    rplus(y)*x
  else
    stop("undefined combination of arguments for *.rcomp")
}

"/.rcomp" <- function(x,y) {
  if( is.rcomp(x) && is.rcomp(y) )
    rcomp(gsi.div(x,y))
  else if( is.rcomp(x) )
    rplus(x)/y
  else
    stop("undefined combination of arguments for /.rcomp")
}

"+.rplus" <- function(x,y) {
  if( is.rplus(x) && is.rplus(y) )
    rplus(gsi.add(x,y))
  else
    rmult(gsi.add(x,y))
}

"-.rplus" <- function(x,y) {
  if( missing(y) )
    rmult(-unclass(x))
  else
    rmult(gsi.sub(x,y))
}


"*.rplus" <- function(x,y) {
  if( is.rplus(x) && is.rplus(y) )
    rplus(gsi.mul(x,y))
  else if( is.rplus(x) )
    mul.rplus(x,y)
  else if( is.rplus(y) )
    mul.rplus(y,x)
  else
    stop("undefined combination of arguments for *.rplus")
}

"/.rplus" <- function(x,y) {
  if( is.rplus(x) && is.rplus(y) )
    rplus(gsi.div(x,y))
  else if( is.rcomp(x) )
    mul.rplus(rplus(x),1/unclass(y))
  else
    stop("undefined combination of arguments for /.rcomp")
}

"+.rmult" <- function(x,y) {
  rmult(gsi.add(x,y))
}

"-.rmult" <- function(x,y) {
  if( missing(y) )
    rmult(-unclass(x))
  else
    rmult(gsi.sub(x,y))
}


"*.rmult" <- function(x,y) {
  if( is.rmult(x) && is.rmult(y) )
    rmult(gsi.mul(x,y))
  else
    rmult(unclass(x)*unclass(y))
}

"/.rmult" <- function(x,y) {
  if( is.rmult(x) && is.rmult(y) )
    rmult(gsi.div(x,y))
  else 
    rmult(unclass(x)/unclass(y))
}

"%*%" <- function(x,y) UseMethod("%*%",structure(c(),class=c(class(x),class(y))))


#gsi.internaltmp <- get("%*%",pos="package:base")
#formals(gsi.internaltmp) <- formals(get("%*%"))
#"%*%.default" <- gsi.internaltmp

"%*%.default" <- function(x,y) base::"%*%"(x,y)

"%*%.rmult" <- function(x,y) {
  if( is.rmult(y) )
    if( is.rmult(x) ) 
      c(gsi.mul(x,y) %*% rep(1,gsi.getD(x)))
    else if( is.matrix(x) ) 
      rmult(gsi.simshape(oneOrDataset(y) %*% t(x),y))
    else
      c(oneOrDataset(y) %*% x) 
  else if( is.matrix(y) )
      rmult(gsi.simshape(oneOrDataset(x) %*% y,x))
  else
      c(oneOrDataset(x) %*% y) 
  }

"%*%.acomp" <- function(x,y) {
  if( is.acomp(y) )
    if( is.acomp(x) ) 
      cdt(x) %*% cdt(y)
    else if( is.matrix(x) ) {
      if( nrow(x) == gsi.getD(y) )
        clr.inv(x %*% clr(y))
      else
        ilr.inv(x %*% ilr(y))
    }
    else
      stop( "%*%.acomp is only defined for special combinations I" )
  else if( is.acomp(x) ) {
    if( is.matrix(y) ) {
      if( ncol(y) == gsi.getD(x) )
        clr.inv(clr(x) %*% y )
      else
        ilr.inv(ilr(x) %*% y )
    }
  else
      stop( "%*%.acomp is only defined for special combinations II" )
  }
  else
      stop( "%*%.acomp is only defined for special combinations III" )
    
}

"%*%.aplus" <- function(x,y) {
  if( is.aplus(y) )
    if( is.aplus(x) ) 
      cdt(x) %*% cdt(y)
    else if( is.matrix(x) ) {
        ilt.inv(x %*% ilt(y))
    }
    else
      stop( "%*%.acomp is only defined for special combinations I" )
  else if( is.aplus(x) ) {
    if( is.matrix(y) ) {
        ilt.inv(ilt(x) %*% y )
    }
  else
      stop( "%*%.aplus is only defined for special combinations II" )
  }
  else
      stop( "%*%.aplus is only defined for special combinations III" )
    
}


convex.rcomp <- function(x,y,alpha=0.5) {
  rcomp( alpha*x + (1-alpha)*y )
}


mul.rplus <- function(x,r) {
  if( all(r>=0) )
    rplus(unclass(x)*r)
  else
    rmult(unclass(x)*r)
}

power.aplus <- function(x,r) {
  aplus(unclass(x)^r) 
}


gsi.expandrcomp <- function(x,alpha) {
  cpt.inv(cpt(x)*alpha)
}


scale.acomp <- function( x,center=TRUE, scale=TRUE ) {
  W <- x
  if( center ) {
    W <- clr.inv( scale(clr(W),center=center,scale=FALSE) )
    if( scale )
      W <- power.acomp(W,as.numeric(scale)/
                       sqrt(sum(gsi.diagExtract(var(clr(W))))))
  } else if( scale ) {
    mean <- c(mean.acomp(W))
    W <- perturbe(power.acomp(perturbe(W,1/mean),as.numeric(scale)/sqrt(sum(gsi.diagExtract(var(clr(W)))))),mean)
  }
  W
}

scale.rcomp <- function( x,center=TRUE, scale=TRUE ) {
  W <- x
  if( center ) {
    W <- cpt.inv( scale(cpt(W),center=center,scale=FALSE) )
    if( scale )
      W <- gsi.expandrcomp(W,as.numeric(scale)/sqrt(sum(gsi.diagExtract(var(cpt(W))))))
  } else if( scale ) {
    mean <- c(mean.rcomp(W))
    W <- gsi.add(mean,gsi.sub(W,mean)/sqrt(sum(gsi.diagExtract(var(cpt(W))))))
  }
  W
}

scale.aplus <- function( x,center=TRUE, scale=TRUE ) {
  W <- x
  if( center ) {
    W <- ilt.inv( scale(ilt(W),center=center,scale=FALSE) )
    if( scale )
      W <- power.aplus(W,as.numeric(scale)/sqrt(sum(gsi.diagExtract(var(ilt(W))))))
  } else if( scale ) {
    mean <- c(mean.aplus(W))
    W <- perturbe.aplus(power.aplus(perturbe.aplus(W,1/mean),as.numeric(scale)/sqrt(sum(gsi.diagExtract(var(ilt(W)))))),mean)
  }
  W
}

scale.rplus <- function( x,center=TRUE, scale=TRUE ) {
   rmult(scale(gsi.plain(x),center=center,scale=scale))
}

scale.rmult <- function( x,center=TRUE, scale=TRUE ) {
   rmult(scale(gsi.plain(x),center=center,scale=scale))
}

normalize <- function(x,...) UseMethod("normalize",x)
normalize.default <- function(x,...) x/norm(x)

norm <- function(x,...) UseMethod("norm",x)

norm.default <- function(x,...) {
  sqrt( sum(x^2) )
}

norm.acomp <- function(x,...) {
  norm.rmult(cdt(x),...)
}
norm.rcomp <- norm.acomp
norm.aplus <- norm.acomp
norm.rplus <- norm.acomp
norm.rmult <- function(x,...) {
   sqrt(x %*% x)
}

dist <- function(x,...) UseMethod("dist")
dist.default <- function(x,...) mva::dist(cdt(x),...)


scalar <- function(x,y) UseMethod("scalar")

scalar.default <- function(x,y) {
  x <- cdt(x)
  y <- cdt(y)
  tmp <- gsi.mul(oneOrDataset(x,y), oneOrDataset(y,x)) 
  c( tmp %*% rep(1,NCOL(tmp)))
}


clr <- function( x ) {
  W <- oneOrDataset(x)
  rmult(gsi.simshape(unclass(log( W / c(geometricmean.row(W)))),x)) 
}

clr.inv <- function( z ) {
  acomp( exp(z) )
}

ult <- function( x ) {
  ilt(clo(x))
}

ult.inv <- clr.inv

Kappa <- function( x ) {
  W <- oneOrDataset(x)
  (clr(W)-ult(W))[,1]
}

gsi.ilrBase <- function(D) {
  if( D==1 )
    return(matrix(nrow=0,ncol=0))
  tmp <- diag(D) - 1/(D)* matrix(1,ncol=D,nrow=D)
  for( i in 1:(NCOL(tmp)-1)  ) {
    tmp[,i] <- tmp[,i]/sqrt(sum(tmp[,i]^2))
    rest <- (i+1):NCOL(tmp)
    if( length(rest) != 1 ) {
      tmp[,rest]  <-tmp[,rest,drop=FALSE] - tmp[,rep(i,length(rest)),drop=FALSE]%*%
        gsi.diagGenerate( c(t(tmp[,i])%*%tmp[,rest,drop=FALSE] ) )
    } 
  }
tmp[,-NROW(tmp)]
}

ilrBaseList <- lapply(1:20,gsi.ilrBase)
ilrBase <- function( x=NULL , z=NULL , D = NULL ) {
  if( missing(D) )
    D <- if(is.null(x))
      NCOL(oneOrDataset(z))+1
    else
      NCOL(oneOrDataset(x))
  while( D > length(ilrBaseList) )
    ilrBaseList <<- c(ilrBaseList,gsi.ilrBase(length(ilrBaseList)+1))
  ilrBaseList[[D]]
}

ilr    <- function( x , V=ilrBase(x) ) {
  rmult(clr2ilr( clr(oneOrDataset(x)),V ))
}

ilr.inv <- function( z, V=ilrBase(z=z)) {
  clr.inv( ilr2clr(z,V) )
}

alr <- function( x ) {
  W <- oneOrDataset(x)
  rmult(gsi.simshape( log( unclass(W)[,-NCOL(W)] / c(W[,NCOL(W)]) ) , x))
}

alr.inv <- function( z ) {
  Z <- cbind(oneOrDataset(z),0)
  acomp(gsi.simshape( clo(exp(Z)) , z ))
}


apt <- function( x ) {
  W <- oneOrDataset(x)
  rmult(gsi.simshape( gsi.plain(clo( W )[,-NCOL(W)]) , x))
}

apt.inv <- function( z ) {
  Z <- oneOrDataset(z)
  Z <- cbind(Z, 1 - Z %*% rep(1,NCOL(Z)))
  rcomp(gsi.simshape( Z ,z ))
}

cpt <- function( x ) {
  x <- oneOrDataset(x)
  rmult(clo(x)- 1/NCOL(x))
}

cpt.inv <- function( z ) {
  if( abs(sum(z))>0.0001 )
    warning( "z not closed in cpt.inv")
  rcomp(z + 1/NCOL(oneOrDataset(z)))
}

ipt    <- function( x , V=ilrBase(x)) {
  rmult(clr2ilr(cpt(x),V))
}

ipt.inv <- function( z, V=ilrBase(z=z) ) {
  cpt.inv( ilr2clr(z,V) )
}

ilt <- function( x ) {
  rmult(log(gsi.plain(x)))
}

ilt.inv <- function( z ) {
  aplus(exp(z))
}

iit <- function( x ) {
  rmult( x )
}

iit.inv <- function(z) {
  rplus(z)
}

idt         <- function(x) UseMethod("idt",x)
idt.default <- function(x) x
idt.acomp   <- function(x) ilr(x) 
idt.rcomp   <- function(x) ipt(x) 
idt.aplus   <- ilt 
idt.rplus   <- iit 
idt.rmult   <- function(x) x
idt.factor  <- function(x) rmult(clr2ilr(cdt(factor)))


cdt         <- function(x) UseMethod("cdt",x)
cdt.default <- function(x) x
cdt.acomp   <- clr 
cdt.rcomp   <- cpt 
cdt.aplus   <- ilt 
cdt.rplus   <- iit 
cdt.rmult   <- function(x) x
cdt.factor  <- function(x) {
  #x <- matrix(0,nrow=length(x),ncol=nlevels(x),dimnames=list(names(x),levels(x)))
  x[1:ncol(x)+unclass(x)] <- model.matrix(~-1+x)
  
  rmult(matrix(x,nrow=nrow(x),dimnames=dimnames(x)))
}

cor <- function(x,y=NULL,...) UseMethod("cor",x)
cor.default <- base::cor
formals(cor.default) <- c(formals(cor.default),alist(...= ))

cor.acomp <- function(x,y=NULL,...) {
  cr <- cor(cdt(x),cdt(y),...)
}

cor.rcomp <- cor.acomp 
cor.aplus <- cor.acomp
cor.rplus <- cor.acomp
cor.rmult <- function(x,y=NULL,...) cor(unclass(x),unclass(cdt(y)),...)


variation <- function( x, ... ) UseMethod("variation",x)

variation.acomp <- function( x,... ) {
  co <-var(clr(x))
  d <- NCOL(x)
  va <-gsi.diagExtract(co)
  co1 <- matrix(rep(va,each=d),ncol=d)
  co2 <- matrix(rep(va,d),ncol=d)
  -2*co+co1+co2
  
}

variation.rcomp <- function( x ,...) {
  co <-var(cpt(x))
  d <- NCOL(x)
  va <-gsi.diagExtract(co)
  co1 <- matrix(rep(va,each=d),ncol=d)
  co2 <- matrix(rep(va,d),ncol=d)
  -2*co+co1+co2
  
}


variation.aplus <- function( x ,...) {
  co <-var(ilt(x))
  d <- NCOL(x)
  va <-gsi.diagExtract(co)
  co1 <- matrix(rep(va,each=d),ncol=d)
  co2 <- matrix(rep(va,d),ncol=d)
  -2*co+co1+co2
  
}

variation.rmult <- function( x ,...) {
  co <-var(iit(x))
  d <- NCOL(x)
  va <-gsi.diagExtract(co)
  co1 <- matrix(rep(va,each=d),ncol=d)
  co2 <- matrix(rep(va,d),ncol=d)
  -2*co+co1+co2
  
}
variation.rplus <- variation.rmult

covariation <- function(X,...) UseMethod("covariation",X)

covariation.acomp <- function( X ,...) {
  i <- rep(1:NCOL(X),each=NCOL(X))
  j <- rep(1:NCOL(X),NCOL(X))
  take <- i<j
  TM <- matrix(0,ncol=sum(take),nrow=NCOL(X))
  TM[i[take]+NROW(TM)*((1:sum(take))-1)] <- 1
  TM[j[take]+NROW(TM)*((1:sum(take))-1)] <- -1
  dim(TM) <- c(NCOL(X),sum(take))
  colnames(TM) <- paste( colnames(X)[i[take]],colnames(X)[j[take]],sep="")
  
  t(TM) %*% var(clr(X)) %*% TM 
}

covariation.rcomp <- function( X ,...) {
  i <- rep(1:NCOL(X),each=NCOL(X))
  j <- rep(1:NCOL(X),NCOL(X))
  take <- i<j
  TM <- matrix(0,ncol=sum(take),nrow=NCOL(X))
  TM[i[take]+NROW(TM)*((1:sum(take))-1)] <- 1
  TM[j[take]+NROW(TM)*((1:sum(take))-1)] <- -1
  dim(TM) <- c(NCOL(X),sum(take))
  colnames(TM) <- paste( colnames(X)[i[take]],colnames(X)[j[take]],sep="")
  
  t(TM) %*% var(ipt(X)) %*% TM 
}

covariation.aplus <- function( X ,...) {
  i <- rep(1:NCOL(X),each=NCOL(X))
  j <- rep(1:NCOL(X),NCOL(X))
  take <- i<j
  TM <- matrix(0,ncol=sum(take),nrow=NCOL(X))
  TM[i[take]+NROW(TM)*((1:sum(take))-1)] <- 1
  TM[j[take]+NROW(TM)*((1:sum(take))-1)] <- -1
  dim(TM) <- c(NCOL(X),sum(take))
  colnames(TM) <- paste( colnames(X)[i[take]],colnames(X)[j[take]],sep="")
  
  t(TM) %*% var(ilt(X)) %*% TM 
}

covariation.rmult <- function( X ,...) {
  i <- rep(1:NCOL(X),each=NCOL(X))
  j <- rep(1:NCOL(X),NCOL(X))
  take <- i<j
  TM <- matrix(0,ncol=sum(take),nrow=NCOL(X))
  TM[i[take]+NROW(TM)*((1:sum(take))-1)] <- 1
  TM[j[take]+NROW(TM)*((1:sum(take))-1)] <- -1
  dim(TM) <- c(NCOL(X),sum(take))
  colnames(TM) <- paste( colnames(X)[i[take]],colnames(X)[j[take]],sep="")
  
  t(TM) %*% var(iit(X)) %*% TM 
}
covariation.rplus <- covariation.rmult




gsi.mapin01 <- function(i,min=0,max=1) {c(min,min+(max-min)/i,max)}
gsi.mapfrom01 <- function(x) {(x[3]-x[1])/(x[2]-x[1])}
gsi.mapmin <- function(x) {x[1]}
gsi.mapmax <- function(x) {x[3]}

plot.acomp <- function(x,...,labels=colnames(X),cn=colnames(X),aspanel=FALSE,id=FALSE,idlabs=NULL,idcol=2,center=FALSE,scale=FALSE,pca=FALSE,col.pca=par("col"),margin="acomp",add=FALSE,triangle=!add,col=par("col")) {
  col <- unclass(col)
  X <- oneOrDataset(x)
  s60 <- sin(pi/3)
  c60 <- cos(pi/3)
  if( NCOL(X) > 3 ) {
    if( margin=="rcomp" )
      infkt <- function(x,y,...) {
        plot.acomp(rcompmargin(X,d=c(gsi.mapfrom01(x),gsi.mapfrom01(y)),pos=1)[,c(3,2,1)],...,aspanel=TRUE,center=center,scale=scale,pca=pca,col.pca=col.pca,col=col)
      }
    else if(margin=="acomp") {
      infkt <- function(x,y,...) {
        plot.acomp(acompmargin(X,d=c(gsi.mapfrom01(x),gsi.mapfrom01(y)),pos=1)[,c(3,2,1)],...,aspanel=TRUE,center=center,scale=scale,pca=pca,col.pca=col.pca,col=col)
      }
      
    } else {
      if( !is.numeric(margin))
        margin <- match(margin,colnames(X))
      fest <- X[,margin,drop=FALSE]
      X    <- X[,-margin]
      infkt <- function(x,y,...) {
        plot.acomp(acomp(cbind(X[,c(gsi.mapfrom01(y),gsi.mapfrom01(x))],fest)),...,aspanel=TRUE,center=center,scale=scale,pca=pca,col.pca=col.pca,col=col)
      }
    }
    nn <- NCOL(X)
    pairs(sapply(1:NCOL(X),gsi.mapin01),labels=labels,panel=infkt,...)
    return(invisible(NULL));
  }
  if( is.null(cn) ) {
    cn <- c("x","y","z")
  }
  
  
  if( aspanel ) {
    usr <- par("usr"); on.exit(par(usr))
    par( usr=c(0,1,0,1) )
    lines(x=c(0,c60,1,0),y=c(0,s60,0,0))
    text(0,0.2,cn[1],pos=4,offset=0.01,xpd=TRUE)
    text(1,0.2,cn[2],pos=2,offset=0.01,xpd=TRUE)
    text(0.5,s60,cn[3],pos=3,offset=0.01,xpd=TRUE)
  } else {
    if( !add ) {
      plot(x=c(0,c60,1,0),y=c(0,s60,0,0),
           xlim=c(0,1),ylim=c(0,1),type="n",xlab="",ylab="",
           axes=FALSE)
    }
  if( triangle ) {
    segments(x0=c(0,1,c60),y0=c(0,0,s60),x1=c(1,c60,0),y1=c(0,s60,0))
    axis(1,pos=0)
    mtext(cn[1],side=1,adj=0)
    mtext(cn[2],side=1,adj=1)
    text(0.5,s60,cn[3],pos=3,offset=0.01,xpd=TRUE)
  }
  }
  X <- acomp(X,c(1,2,3))
  Y <- scale.acomp(X,center=center,scale=scale)
  x <- Y[,2]+Y[,3]*c60
  y <- Y[,3]*s60
  points(x,y,...)
  if( pca ) {
    pca.d <- clr.inv(princomp(clr(Y))$loadings[,1])
    pca.c <- mean.acomp(Y)
    lines.acomp(pca.c,pca.d,col=col.pca)
  }
  if( id ) {
    if( is.null(idlabs) )
      idlabs <- paste(cn[1],"=",round(X[,1],2),",\n",
                      cn[2],"=",round(X[,2],2),",\n",
                    cn[3],"=",round(X[,3],2))
    return( identify(x,y,idlabs,col=idcol,xpd=NA))
  }
  return( invisible(NULL))
}


plot.rcomp <- function(x,...,labels=colnames(X),cn=colnames(X),aspanel=FALSE,id=FALSE,idlabs=NULL,idcol=2,center=FALSE,scale=FALSE,pca=FALSE,col.pca=par("col"),margin="rcomp",add=FALSE) {
X <- oneOrDataset(x)
s60 <- sin(pi/3)
c60 <- cos(pi/3)
if( NCOL(X) > 3 ) {
  if( margin=="rcomp" )
    infkt <- function(x,y,...) {
      plot.rcomp(rcompmargin(X,d=c(gsi.mapfrom01(x),gsi.mapfrom01(y)),pos=1)[,c(3,2,1)],...,aspanel=TRUE,center=center,scale=scale,pca=pca,col.pca=col.pca)
    }
  else if(margin=="acomp") {
    infkt <- function(x,y,...) {
      plot.rcomp(acompmargin(X,d=c(gsi.mapfrom01(x),gsi.mapfrom01(y)),pos=1)[,c(3,2,1)],...,aspanel=TRUE,center=center,scale=scale,pca=pca,col.pca=col.pca)
    }
    
  } else {
    if( !is.numeric(margin))
      margin <- match(margin,colnames(X))
    fest <- X[,margin,drop=FALSE]
    X    <- X[,-margin]
    infkt <- function(x,y,...) {
      plot.rcomp(acomp(cbind(X[,c(gsi.mapfrom01(y),gsi.mapfrom01(x))],fest)),...,aspanel=TRUE,center=center,scale=scale,pca=pca,col.pca=col.pca)
    }
  }
  pairs(sapply(1:NCOL(X),gsi.mapin01),labels=labels,panel=infkt,...)
  return(invisible(NULL));
  }
if( is.null(cn) ) {
  cn <- c("x","y","z")
}
X <- rcomp(X,c(1,2,3))
Y <- scale.rcomp(X,center=center,scale=scale)
x <- Y[,2]+Y[,3]*c60
y <- Y[,3]*s60

if( aspanel ) {
  usr <- par("usr"); on.exit(par(usr))
  par( usr=c(0,1,0,1) )
  lines(x=c(0,c60,1,0),y=c(0,s60,0,0))
  text(0,0.2,cn[1],pos=4,offset=0.01,xpd=TRUE)
  text(1,0.2,cn[2],pos=2,offset=0.01,xpd=TRUE)
  text(0.5,s60,cn[3],pos=3,offset=0.01,xpd=TRUE)

} else {
  if( !add ) {
    eqscplot(x=c(0,c60,1,0),y=c(0,s60,0,0),
             xlim=c(min(c(0,x)),max(c(1,x))),
             ylim=c(min(c(0,y)),max(c(1,y))),
             type="n",xlab="",ylab="",
             axes=FALSE)
    arrows(x0=c(0,1,c60),y0=c(0,0,s60),x1=c(1,c60,0),y1=c(0,s60,0),angle=15)
    axis(1,pos=0)
    mtext(cn[1],side=1,adj=0)
    mtext(cn[2],side=1,adj=1)
    text(0.5,s60,cn[3],pos=3,offset=0.01,xpd=TRUE)
  }
}
points(x,y,...)
if( pca ) {
  pca.d <- cpt.inv(princomp(cpt(Y))$loadings[,1])
  pca.c <- mean.rcomp(Y)
  lines.rcomp(pca.c,pca.d,col=col.pca)
}
if( id ) {
  if( is.null(idlabs) )
    idlabs <- paste(cn[1],"=",round(X[,1],2),",\n",
                    cn[2],"=",round(X[,2],2),",\n",
                    cn[3],"=",round(X[,3],2))
  return( identify(x,y,idlabs,col=idcol,xpd=NA))
}
return( invisible(NULL))
}

plot.aplus <- function(x,...,labels=colnames(X),cn=colnames(X),aspanel=FALSE,id=FALSE,idlabs=NULL,idcol=2,center=FALSE,scale=FALSE,pca=FALSE,col.pca=par("col"),add=FALSE,logscale=TRUE) {
X <- oneOrDataset(x)
if( NCOL(X) > 2 ) {
    infkt <- function(x,y,...) {
      plot.aplus(X[,c(x[1],y[1])],...,aspanel=TRUE,center=center,scale=scale,pca=pca,col.pca=col.pca,logscale=logscale)
    }
    pairs(matrix(1:NCOL(X),nrow=1),labels=labels,panel=infkt,log="xy",...)
    return(invisible(NULL));
  }
if( is.null(cn) ) {
  cn <- c("x","y")
}
x <- X[,1]
y <- X[,2]
if( aspanel ) {
  usr <- par("usr"); on.exit(par(usr))
  if( logscale )
    par( xlog=TRUE,ylog=TRUE,usr=c(log10(min(x)),log10(max(x)),log10(min(y)),log10(max(y))))
  else
    par( usr=c(min(x),max(x),min(y),max(y)) )
  #axis(1)
  #axis(2)
} else {
  if( !add ) {
    plot(x=c(1),y=c(1),
         xlim=range(x),ylim=range(y),type="n",
         log=ifelse(logscale,"xy",""),xlab=cn[1],ylab=cn[2])
  }
}
points(x,y,...)
if( pca ) {
  pca.d <- ilt.inv(princomp(ilt(X))$loadings[,1])
  pca.c <- mean.aplus(X)
  lines.aplus(pca.c,pca.d,col=col.pca)
}
if( id ) {
  if( is.null(idlabs) )
    idlabs <- paste(cn[1],"=",round(X[,1],2),",\n",
                    cn[2],"=",round(X[,2],2))
  return( identify(x,y,idlabs,col=idcol,xpd=NA))
}
return( invisible(NULL))
}

plot.rplus <- function(x,...,labels=colnames(X),cn=colnames(X),aspanel=FALSE,id=FALSE,idlabs=NULL,idcol=2,center=FALSE,scale=FALSE,pca=FALSE,col.pca=par("col"),add=FALSE,logscale=FALSE) {
X <- oneOrDataset(x)
if( NCOL(X) > 2 ) {
    infkt <- function(x,y,...) {
      plot.rplus(X[,c(x[1],y[1])],...,aspanel=TRUE,center=center,scale=scale,pca=pca,col.pca=col.pca,logscale=logscale)
    }
    pairs(matrix(1:NCOL(X),nrow=1),labels=labels,panel=infkt,...)
    return(invisible(NULL));
  }
if( is.null(cn) ) {
  cn <- c("x","y")
}
x <- X[,1]
y <- X[,2]
if( aspanel ) {
  usr <- par("usr"); on.exit(par(usr))
  if( logscale )
    par( xlog=TRUE,ylog=TRUE,usr=c(log10(min(x)),log10(max(x)),log10(min(y)),log10(max(y))))
  else
    par( usr=c(min(x),max(x),min(y),max(y)) )
  #axis(1)
  #axis(2)
} else {
  if( !add ) {
    plot(x=c(1),y=c(1),
         xlim=range(x),ylim=range(y),type="n",
         log=ifelse(logscale,"xy",""),xlab=cn[1],ylab=cn[2])
  }
}
points(x,y,...)
if( pca ) {
  pca.d <- iit.inv(princomp(iit(X))$loadings[,1])
  pca.c <- mean.rplus(X)
  lines.rplus(pca.c,pca.d,col=col.pca)
}
if( id ) {
  if( is.null(idlabs) )
    idlabs <- paste(cn[1],"=",round(X[,1],2),",\n",
                    cn[2],"=",round(X[,2],2))
  return( identify(x,y,idlabs,col=idcol,xpd=NA))
}
return( invisible(NULL))
}

plot.rmult <- function(x,...,labels=colnames(X),cn=colnames(X),aspanel=FALSE,id=FALSE,idlabs=NULL,idcol=2,center=FALSE,scale=FALSE,pca=FALSE,col.pca=par("col"),add=FALSE,logscale=FALSE) {
X <- oneOrDataset(x)
if( NCOL(X) > 2 ) {
    infkt <- function(x,y,...) {
      plot.rmult(X[,c(x[1],y[1])],...,aspanel=TRUE,center=center,scale=scale,pca=pca,col.pca=col.pca,logscale=logscale)
    }
    pairs(matrix(1:NCOL(X),nrow=1),labels=labels,panel=infkt,...)
    return(invisible(NULL));
  }
if( is.null(cn) ) {
  cn <- c("x","y")
}
x <- X[,1]
y <- X[,2]
if( aspanel ) {
  usr <- par("usr"); on.exit(par(usr))
  if( logscale )
    par( xlog=TRUE,ylog=TRUE,usr=c(log10(min(x)),log10(max(x)),log10(min(y)),log10(max(y))))
  else
    par( usr=c(min(x),max(x),min(y),max(y)) )
  #axis(1)
  #axis(2)
} else {
  if( !add ) {
    plot(x=c(1),y=c(1),
         xlim=range(x),ylim=range(y),type="n",
         log=ifelse(logscale,"xy",""),xlab=cn[1],ylab=cn[2])
  }
}
points(x,y,...)
if( pca ) {
  pca.d <- iit.inv(princomp(iit(X))$loadings[,1])
  pca.c <- mean(X)
  lines.rmult(pca.c,pca.d,col=col.pca)
}
if( id ) {
  if( is.null(idlabs) )
    idlabs <- paste(cn[1],"=",round(X[,1],2),",\n",
                    cn[2],"=",round(X[,2],2))
  return( identify(x,y,idlabs,col=idcol,xpd=NA))
}
return( invisible(NULL))
}


line.acomp <- function(X,...,steps=30) {
  X <- oneOrDataset(X)
  Y <- X[-1,,drop=FALSE]
  X <- X[-nrow(X),,drop=FALSE]
  s60 <- sin(pi/3)
  c60 <- cos(pi/3)
  l   <- rep((0:steps)/steps,NROW(X))
  i   <- rep(1:NROW(X),each=steps+1)
  XP  <- ilr.inv((1-l)*ilr(X[i,,drop=FALSE]) + l*ilr(Y[i,,drop=FALSE]))
  x <- XP[,2]+XP[,3]*c60
  y <- XP[,3]*s60
  lines(x,y,...)
}

line.rcomp <- function(X,...,whole=FALSE,steps=30) {
  X <- oneOrDataset(X)
  Y <- X[-1,,drop=FALSE]
  X <- X[-nrow(X),,drop=FALSE]
  s60 <- sin(pi/3)
  c60 <- cos(pi/3)
  l   <- rep(c((0:steps)/steps),NROW(X))
  i   <- rep(1:NROW(X),each=steps+1)
  XP  <- convex.rcomp(X[i,,drop=FALSE],Y[i,,drop=FALSE],l)
  x <- XP[,2]+XP[,3]*c60
  y <- XP[,3]*s60
  lines(x,y,...)
}


line.aplus <- function(X,...,steps=30) {
  X <- oneOrDataset(X)
  Y <- X[-1,,drop=FALSE]
  X <- X[-nrow(X),,drop=FALSE]
  l   <- rep((0:steps)/steps,NROW(X))
  i   <- rep(1:NROW(X),each=steps+1)
  XP  <- ilt.inv((1-l)*ilt(X[i,,drop=FALSE]) + l*ilt(Y[i,,drop=FALSE]))
  x <- XP[,1]
  y <- XP[,2]
  lines(x,y,...)
}

line.rplus <- function(X,...,steps=30) {
  X <- oneOrDataset(X)
  Y <- X[-1,,drop=FALSE]
  X <- X[-nrow(X),,drop=FALSE]
  l   <- rep((0:steps)/steps,NROW(X))
  i   <- rep(1:NROW(X),each=steps+1)
  XP  <- iit.inv((1-l)*iit(X[i,,drop=FALSE]) + l*iit(Y[i,,drop=FALSE]))
  x <- XP[,1]
  y <- XP[,2]
  lines(x,y,...)
}

line.rmult <- function(X,...,steps=30) {
  X <- oneOrDataset(X)
  Y <- X[-1,,drop=FALSE]
  X <- X[-nrow(X),,drop=FALSE]
  l   <- rep((0:steps)/steps,NROW(X))
  i   <- rep(1:NROW(X),each=steps+1)
  XP  <- (1-l)*unclass(X)[i,,drop=FALSE] + l*unclass(Y)[i,,drop=FALSE]
  x <- XP[,1]
  y <- XP[,2]
  lines(x,y,...)
}


segments.acomp <- function(X,Y,...,whole=FALSE,steps=30) {
  X <- oneOrDataset(X,Y)
  Y <- oneOrDataset(Y,X)
  s60 <- sin(pi/3)
  c60 <- cos(pi/3)
  l   <- rep(c((0:steps)/steps,NA),NROW(X))
  i   <- rep(1:NROW(X),each=steps+2)
  XP  <- ilr.inv((1-l)*ilr(X[i,]) + l*ilr(Y[i,]))
  x <- XP[,2]+XP[,3]*c60
  y <- XP[,3]*s60
  lines(x,y,...)
}

segments.rcomp <- function(X,Y,...,whole=FALSE,steps=1) {
  X <- oneOrDataset(X,Y)
  Y <- oneOrDataset(Y,X)
  s60 <- sin(pi/3)
  c60 <- cos(pi/3)
  l   <- rep(c((0:steps)/steps,NA),NROW(X))
  i   <- rep(1:NROW(X),each=steps+2)
  XP  <- convex.rcomp(X[i,],Y[i,],l)
  x <- XP[,2]+XP[,3]*c60
  y <- XP[,3]*s60
  lines(x,y,...)
}

segments.aplus <- function(X,Y,...,whole=FALSE,steps=30) {
  X <- oneOrDataset(X,Y)
  Y <- oneOrDataset(Y,X)
  l   <- rep(c((0:steps)/steps,NA),NROW(X))
  i   <- rep(1:NROW(X),each=steps+2)
  XP  <- ilt.inv((1-l)*ilt(X[i,]) + l*ilt(Y[i,]))
  x <- XP[,1]
  y <- XP[,2]
  lines(x,y,...)
}

segments.rplus <- function(X,Y,...,whole=FALSE,steps=30) {
  X <- oneOrDataset(X,Y)
  Y <- oneOrDataset(Y,X)
  l   <- rep(c((0:steps)/steps,NA),NROW(X))
  i   <- rep(1:NROW(X),each=steps+2)
  XP  <- iit.inv((1-l)*iit(X[i,]) + l*iit(Y[i,]))
  x <- XP[,1]
  y <- XP[,2]
  lines(x,y,...)
}

segments.rmult <- function(X,Y,...,whole=FALSE,steps=30) {
  X <- oneOrDataset(X,Y)
  Y <- oneOrDataset(Y,X)
  l   <- rep(c((0:steps)/steps,NA),NROW(X))
  i   <- rep(1:NROW(X),each=steps+2)
  XP  <- (1-l)*unclass(X)[i,] + l*unclass(Y)[i,]
  x <- XP[,1]
  y <- XP[,2]
  lines(x,y,...)
}



segments.panel.acomp <- function(X,Y,...,whole=FALSE,steps=30) {
  function(what,...) {
    if( !is.null(colnames(what)) )
      what <- colnames(what)
    X <- gsi.margin(X,what)
    Y <- gsi.margin(Y,what)
    segments.acomp(X,Y,...,whole=whole,steps=steps)
  }
}

segments.panel.rcomp <- function(X,Y,...,whole=FALSE,steps=30) {
  function(what,...) {
    if( !is.null(colnames(what)) )
      what <- colnames(what)
    X <- gsi.margin(X,what)
    Y <- gsi.margin(Y,what)
    segments.rcomp(X,Y,...,whole=whole,steps=steps)
  }
}

segments.panel.aplus <- function(X,Y,...,whole=FALSE,steps=30) {
  function(what,...) {
    if( !is.null(colnames(what)) )
      what <- colnames(what)
    X <- X[,what]
    Y <- Y[,what]
    segments.aplus(X,Y,...,whole=whole,steps=steps)
  }
}

segments.panel.rplus <- function(X,Y,...,whole=FALSE,steps=30) {
  function(what,...) {
    if( !is.null(colnames(what)) )
      what <- colnames(what)
    X <- X[,what]
    Y <- Y[,what]
    segments.rplus(X,Y,...,whole=whole,steps=steps)
  }
}

segments.panel.rmult <- function(X,Y,...,whole=FALSE,steps=30) {
  function(what,...) {
    if( !is.null(colnames(what)) )
      what <- colnames(what)
    X <- X[,what]
    Y <- Y[,what]
    segments.rplus(X,Y,...,whole=whole,steps=steps)
  }
}



gsi.closespread <- function(spread) {
  if(length(dim(spread))>3) {
    return(apply(spread,1,gsi.closespread))
  }
  d <- nrow(spread)
  Pmat <- diag(d)-1/d
  row.names(Pmat)<-row.names(spread)
  Pmat %*% spread %*% t(Pmat)
}

gsi.spreadToIsoSpace <- function(spread) {
  if(length(dim(spread))>3) {
    return(apply(spread,1,gsi.closespread))
  }
  d <- nrow(spread)
  V <- ilrBase(D=d)
  t(V) %*% spread %*% V
}

ellipses.rcomp <- function(mid,spread,t=1,...,steps=360) {
mid <- ipt(mid)
sp <- spread
w  <- seq(0,2*pi,length.out=steps)
for(i in 1:nrow(mid)) {
    if( length(dim(spread))==3 )
      sp<-spread[i,,]
    isp <- gsi.spreadToIsoSpace(sp)
    mi <- mid[i,]
    eisp <- eigen(isp,T)
    X <- t(mi+ t(sqrt(eisp$values[1])*t*cos(w) %o% eisp$vectors[,1] +
             sqrt(eisp$values[2])*t*sin(w) %o% eisp$vectors[,2])) 
    line.rcomp(ipt.inv(X),...)
  }
}

ellipses.acomp <- function(mid,spread,t=1,...,steps=360) {
mid <- ilr(mid)
sp <- spread
w  <- seq(0,2*pi,length.out=steps)
for(i in 1:nrow(mid)) {
    if( length(dim(spread))==3 )
      sp<-spread[i,,]
    isp <- gsi.spreadToIsoSpace(sp)
    mi <- mid[i,]
    eisp <- eigen(isp,T)
    X <- t(mi+ t(sqrt(eisp$values[1])*t*cos(w) %o% eisp$vectors[,1] +
             sqrt(eisp$values[2])*t*sin(w) %o% eisp$vectors[,2])) 
    line.rcomp(ilr.inv(X),...)
  }
}




lines.acomp <- function(x,d,...,whole=FALSE,steps=30) {
  X <- oneOrDataset(x,d)
  d <- oneOrDataset(d,x)
  d <- normalize(acomp(d)) 
  X <- perturbe(X,power.acomp(d,-scalar(acomp(X),acomp(d)))) 
  s60 <- sin(pi/3)
  c60 <- cos(pi/3)
  l   <- rep(2*c((0:steps)/steps,NA)-1,NROW(X))
  i   <- rep(1:NROW(X),each=steps+2)
  XP  <- acomp(clr.inv(clr(X[i,]) + (2*l)^3*clr(d[i,])))
  x <- XP[,2]+XP[,3]*c60
  y <- XP[,3]*s60
  lines(x,y,...)
}



lines.rcomp <- function(x,d,...,whole=FALSE,steps=30) {
  X <- oneOrDataset(x,d)
  d <- oneOrDataset(d,x)
  l1 <- apply(-d/X,1,function(x) {max(c(0,x[x<0]))}) 
  l2 <- apply(-d/X,1,function(x) {min(c(0,x[x>0]))})
  X1 <- rcomp(gsi.add(X,l1*d)) 
  X2 <- rcomp(gsi.add(X,l2*d))
  s60 <- sin(pi/3)
  c60 <- cos(pi/3)
  segments.rcomp(X1,X2,...)
}


lines.panel.acomp <- function(x,d,...,whole=FALSE,steps=30) {
  function(what,...) {
    if( !is.null(colnames(what)) )
      what <- colnames(what)
    x <- margin(x,what)
    d <- margin(d,what)
    lines.acomp(x,d,...,whole=whole,steps=steps)
  }
}

lines.panel.rcomp <- function(x,d,...,whole=FALSE,steps=30) {
  function(what,...) {
    if( !is.null(colnames(what)) )
      what <- colnames(what)
    x <- margin(x,what)
    d <- margin(d,what)
    lines.rcomp(x,d,...,whole=whole,steps=steps)
  }
}



rDirichlet.acomp <- function(n,alpha) {
  acomp(sapply(alpha,rgamma,n=n))
}

rDirichlet.rcomp <- function(n,alpha) {
  rcomp(sapply(alpha,rgamma,n=n))
}


runif.acomp <- function(n,D) rDirichlet.acomp(n,rep(1,D))
runif.rcomp <- function(n,D) rDirichlet.rcomp(n,rep(1,D))

rnorm.aplus <- function(n,mean,var) {
  D <- NCOL(oneOrDataset(mean))
  print(D)
  perturbe.aplus(ilt.inv(matrix(rnorm(n*length(mean)),ncol=D) %*% chol(var)),
                mean)
}

dnorm.aplus <- function(x,mean,var) {
  w <- ilt(perturbe.aplus(x,1/mean))
  D <- ncol(oneOrDataset(x))
  if( length(dim(w)) == 2 ) 
    u <- c(rep(1,ncol(w))%*%((solve(var,t(w)))*t(w)))
  else
    u <- sum(solve(var,w)*w)
  exp(-u/2)/sqrt(2^D*pi^D*det(var))
}


rnorm.acomp <- function(n,mean,var) {
  D <- NCOL(oneOrDataset(mean))
  print(D)
  perturbe.acomp(ilr.inv(matrix(rnorm(n*length(mean)),ncol=D-1) %*%
                         chol(clrvar2ilr(var))),
                mean)
}

dnorm.acomp <- function(x,mean,var) {
  w <- ilr(perturbe.acomp(x,1/mean))
  D <- ncol(oneOrDataset(x))
  if( length(dim(w)) == 2 ) 
    u <- c(rep(1,D-1)%*%((solve(clrvar2ilr(var),t(w)))*t(w)))
  else
    u <- sum(solve(clrvar2ilr(var),w)*w)
  exp(-u/2)/sqrt(2*pi*det(clrvar2ilr(var)))
}



rlnorm.rplus <- function(n,meanlog,varlog) {
  D <- NCOL(oneOrDataset(meanlog))
  print(D)
  rplus(perturbe.aplus(exp(matrix(rnorm(n*length(meanlog)),ncol=D) %*% chol(varlog)),
                exp(meanlog)))
}

dlnorm.rplus <- function(x,meanlog,varlog) {
  xx <- oneOrDataset(x)
  w <- log(perturbe.aplus(x,1/meanlog))
  if( length(dim(w)) == 2 ) {
    u <- c(rep(1,ncol(w))%*%((solve(var,t(w)))*t(w)))
    v <- c(exp(log(xx) %*% rep(1,ncol(xx)))) 
  }
  else {
    u <- solve(var,w)%*%w
    v <- prod(x)
  }
  exp(-u/2)/sqrt(2*pi*det(var))/v
}


rnorm.rplus <- function(n,mean,var) {
  D <- ncol(var)
  rplus(pmax(rplus(matrix(rnorm(n*D),ncol=D) %*% chol(var))+rplus(mean),0))
}

rnorm.rmult <- function(n,mean,var) {
  D <- ncol(var)
  rmult(matrix(rnorm(n*D),ncol=D) %*% chol(var))+rplus(mean)
}

rnorm.rcomp <- function(n,mean,var) {
  D <- ncol(var)
  pmax(ipt.inv(matrix(rnorm(n*D),ncol=D-1) %*% chol(clrvar2ilr(var)))+rplus(mean),0)
}



gsi.margin <- function(X,...) UseMethod("gsi.margin")

gsi.margin.acomp <- function(X,what,...,margin="acomp") {
  if( margin == "sub" )
    acomp(X,what)
  else if( margin=="rcomp" )
      rcompmargin(X,what)
  else if( margin=="acomp")
    acompmargin(X,what)
  else {
    if( !is.numeric(what) )
      what <- match(what,colnames(X))
    if( !is.numeric(margin))
      margin <- match(margin,colnames(X))
    acomp(X,c(what,margin))
  }
}

gsi.margin.rcomp <- function(X,what,...,margin="rcomp") {
  if( margin == "sub" )
    acomp(X,what)
  else if( margin=="rcomp" )
    rcompmargin(X,what)
  else if( margin=="acomp")
    acompmargin(X,what)
  else {
    if( !is.numeric(what) )
      what <- match(what,colnames(X))
    if( !is.numeric(margin))
      margin <- match(margin,colnames(X))
    rcomp(X,c(what,margin))
  }
}

gsi.margin.aplus <- function(X,what,...) {
  aplus(X,what)
}

gsi.margin.rplus <- function(X,what,...) {
  rplus(X,what)
}

gsi.isSingleRow <- function(X) {
  return( NROW(X) == 1 || NCOL(X) ==1 )
}

barplot.acomp <- function(height,...,legend.text=TRUE) {
  X <- height
  if( gsi.isSingleRow(X) )
     barplot(t(rbind(gsi.plain(acomp(X)),0)),c(1,0),...,legend.text=legend.text)
  else
     barplot(gsi.plain(t(acomp(X))),...,legend.text=legend.text)
}

barplot.rcomp <- function(height,...,legend.text=TRUE) {
  X <- height
  if( gsi.isSingleRow(X) )
     barplot(t(rbind(gsi.plain(rcomp(X)),0)),c(1,0),...,legend.text=legend.text)
  else
  barplot(gsi.plain(t(rcomp(X))),...,legend.text=legend.text);
}

barplot.aplus <- function(height,...,legend.text=TRUE,beside=TRUE) {
  X <- height
  if( gsi.isSingleRow(X) )
    barplot(t(rbind(gsi.plain(aplus(X)),0)),c(1,0),...,legend.text=legend.text)
  else
    barplot(gsi.plain(t(aplus(X))),...,legend.text=legend.text,beside=beside);
}

barplot.rplus <- function(height,...,legend.text=TRUE,beside=TRUE) {
  X <- height
  if( gsi.isSingleRow(X) )
    barplot(t(rbind(gsi.plain(rplus(X)),0)),c(1,0),...,legend.text=legend.text)
  else
    barplot(gsi.plain(t(rplus(X))),...,legend.text=legend.text,beside=beside);
}

split.acomp <- function(x,f) {
  cls <- class(x)
  lapply(split(1:NROW(x),f),function(i) structure(x[i,],class=cls))
}
split.rcomp <- split.acomp
split.aplus <- split.acomp
split.rplus <- split.acomp
split.rmult <- split.acomp

as.data.frame.acomp <- function(x,...) as.data.frame.matrix(unclass(x))
as.data.frame.rcomp <- function(x,...) as.data.frame.matrix(unclass(x))
as.data.frame.aplus <- function(x,...) as.data.frame.matrix(unclass(x))
as.data.frame.rplus <- function(x,...) as.data.frame.matrix(unclass(x))
as.data.frame.rmult <- function(x,...) as.data.frame.matrix(unclass(x))

  
